<?php

namespace App\Http\Controllers\Frontend\Auth;

use App\Events\Frontend\Auth\UserLoggedIn;
use App\Events\Frontend\Auth\UserLoggedOut;
use App\Exceptions\GeneralException;
use App\Http\Controllers\Controller;
use App\Services\TenantService;
use Auth;
use Illuminate\Foundation\Auth\AuthenticatesUsers;
use Illuminate\Http\Request;
use LangleyFoxall\LaravelNISTPasswordRules\PasswordRules;
use App\Services\TokenRevocationService;
/**
 * Class LoginController.
 */
class LoginController extends Controller
{
    use AuthenticatesUsers;
    protected $tokenRevocationService;

    public function __construct(TokenRevocationService $tokenRevocationService)
    {
        $this->tokenRevocationService = $tokenRevocationService;
    }

    /**
     * Where to redirect users after login.
     *
     * @return string
     */
    public function redirectPath()
    {
        return route(home_route());
    }

    /**
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function showLoginForm()
    {
        return view('frontend.auth.login');
    }

    /**
     * Get the login username to be used by the controller.
     *
     * @return string
     */
    public function username()
    {
        return config('access.users.username');
    }

    /**
     * Validate the user login request.
     *
     * @param  \Illuminate\Http\Request  $request
     *
     * @throws \Illuminate\Validation\ValidationException
     */
    protected function validateLogin(Request $request)
    {
        $request->validate([
            $this->username() => 'required|string',
            'password' => PasswordRules::login(),
            'g-recaptcha-response' => ['required_if:captcha_status,true', 'captcha'],
        ], [
            'g-recaptcha-response.required_if' => __('validation.required', ['attribute' => 'captcha']),
        ]);
    }

    /**
     * The user has been authenticated.
     *
     * @param Request $request
     * @param         $user
     *
     * @throws GeneralException
     * @return \Illuminate\Http\RedirectResponse
     */
    protected function authenticated(Request $request, $user)
    {
        // Check to see if the users account is confirmed and active
        if (! $user->isConfirmed()) {
            auth()->logout();

            // If the user is pending (account approval is on)
            if ($user->isPending()) {
                throw new GeneralException(__('exceptions.frontend.auth.confirmation.pending'));
            }

            // Otherwise see if they want to resent the confirmation e-mail

            throw new GeneralException(__('exceptions.frontend.auth.confirmation.resend', ['url' => route('frontend.auth.account.confirm.resend', e($user->{$user->getUuidName()}))]));
        }

        if (! $user->isActive()) {
            auth()->logout();

            throw new GeneralException(__('exceptions.frontend.auth.deactivated'));
        }

        event(new UserLoggedIn($user));

        if (config('access.users.single_login')) {
            Auth::logoutOtherDevices($request->password);
        }

        return redirect()->intended($this->redirectPath());
    }

    /**
     * Log the user out of the application.
     *
     * @param Request $request
     *
     * @return \Illuminate\Http\Response
     */
    // public function logout(Request $request)
    // {
    //     // Remove the socialite session variable if exists
    //     if (app('session')->has(config('access.socialite_session_name'))) {
    //         app('session')->forget(config('access.socialite_session_name'));
    //     }

    //     // Fire event, Log out user, Redirect
    //     event(new UserLoggedOut($request->user()));

    //     // Laravel specific logic
    //     $this->guard()->logout();
    //     $request->session()->invalidate();

    //     return redirect()->route('frontend.index');
    // }
    public function logout(Request $request)
    {
        // Get the authenticated user
        $user = $request->user();

        // Get the access token from the session
        $accessToken = $request->session()->get('access_token');

        // Revoke the token from the OAuth provider
        if ($accessToken) {
            $this->tokenRevocationService->revokeToken($accessToken);
        }

        // Remove the refresh token from the database
        $user->microsoft_token = null;
        $user->save();

        // Clear the session if the variables exist
        $request->session()->forget(['access_token', 'access_token_expires', 'refresh_token']);

        // Remove the socialite session variable if exists
        if ($request->session()->has(config('access.socialite_session_name'))) {
            $request->session()->forget(config('access.socialite_session_name'));
        }

        // Fire event, Log out user, Redirect
        event(new UserLoggedOut($user));

        // Laravel specific logic
        $this->guard()->logout();
        $request->session()->invalidate();

        return redirect()->route('frontend.index');
    }

    function loginByApiKey() {
        return TenantService::loginAccessLink();
    }
}
