<?php

namespace App\Repositories\Backend;

use App\Events\Backend\Categories\CategoryCreated;
use App\Events\Backend\Categories\CategoryDeleted;
use App\Events\Backend\Categories\CategoryUpdated;
use App\Exceptions\GeneralException;
use App\Models\Auth\User;
use App\Models\Category;
use App\Models\Log;
use App\Repositories\BaseRepository;
use Illuminate\Support\Str;

class CategoriesRepository extends BaseRepository
{
    /**
     * Associated Repository Model.
     */
    const MODEL = Category::class;

    /**
     * Sortable.
     *
     * @var array
     */
    private $sortable = [
        'id',
        'name',
        'created_at',
        'updated_at',
    ];

    /**
     * Retrieve List.
     *
     * @var array
     * @return Collection
     */
    public function retrieveList(array $options = [])
    {
        $perCategory = isset($options['per_Category']) ? (int) $options['per_Category'] : 20;
        $orderBy = isset($options['order_by']) && in_array($options['order_by'], $this->sortable) ? $options['order_by'] : 'created_at';
        $order = isset($options['order']) && in_array($options['order'], ['asc', 'desc']) ? $options['order'] : 'desc';
        $query = $this->query()
            ->with([
                'owner',
                'updater',
            ])
            ->orderBy($orderBy, $order);

        if ($perCategory == -1) {
            return $query->get();
        }

        return $query->paginate($perCategory);
    }

    /**
     * @return mixed
     */
    public function getForDataTable()
    {
        return $this->query()
        ->leftjoin('users', 'users.id', '=', 'doc_categories.created_by')
        ->leftJoin('doc_categories AS parentcategory', 'parentcategory.id', '=', 'doc_categories.parent_category_id')
        ->select([
            'doc_categories.id',
            'doc_categories.name',
            'doc_categories.url',
            'doc_categories.size',
            'doc_categories.size_s3',
            'doc_categories.created_by',
            'doc_categories.created_at',
            'users.first_name as user_name',
            'parentcategory.name AS parentcategory',
        ]);
    }

    /**
     * @param array $input
     *
     * @throws \App\Exceptions\GeneralException
     *
     * @return bool
     */
    public function create(array $input)
    {

        $input['created_by'] = auth()->user()->id;
        $input['parent_category_id'] =  $input['categories'];
        if ($Category = Category::create($input)) {
            $permissions=Category::FULLPERMISSION;
            $user =User::find(auth()->user()->id);
            if( $user->isGuest())
            {
                $permissions=Category::FULLPERMISSION - Category::SHARE;
            }
            $user->categories()->attach($Category->id, ['permissions' => $permissions]);
            event(new CategoryCreated($Category));
            $log_data = [
                'model'   => 'Category',
                'action'    => 'Create', //Create,Update,Delete
                'user_id' =>  auth()->user()->id,
                'record_id' => $Category->id,
                'label'  => $Category->name,
                'description' => '',
          ];
            Log::create($log_data);
            return $Category->fresh();
        }

        throw new GeneralException(__('exceptions.backend.categories.create_error'));
    }

    /**
     * Update Category.
     *
     * @param \App\Models\Category $Category
     * @param array $input
     */
    public function update(Category $Category, array $input)
    {
        $input['updated_by'] = auth()->user()->id;
        $input['parent_category_id'] =  $input['categories'];
        $logs = [];
        if ($Category->name != $input['name']) {
            $logs[] = [
                'description' => "The Name Change From: " . $Category->name . " To: " . $input['name'],
                'model' => 'Category',
                'action' => 'Update',
                'user_id' =>  auth()->user()->id,
                'record_id' => $Category->id,
                'label' => $Category->name,
            ];
        }
        if ($Category->parent_category_id != $input['parent_category_id']) {
            $parentCategoryName = Category::where('id', $input['parent_category_id'])->value('name');
            $logs[] = [
                'description' => "The Parent Category Change From: " . $Category->parent_category->name . " To: " . $parentCategoryName,
                'model' => 'Category',
                'action' => 'Update',
                'user_id' =>auth()->user()->id,
                'record_id' => $Category->id,
                'label' => $Category->name,
            ];
        }

        if ($Category->update($input)) {
            event(new CategoryUpdated($Category));

            foreach ($logs as $log) {
                Log::create($log);
            }
            return $Category;
        }

        throw new GeneralException(
            __('exceptions.backend.categories.update_error')
        );
    }

    /**
     * @param \App\Models\Category $Category
     *
     * @throws GeneralException
     *
     * @return bool
     */
    public function delete(Category $Category)
    {
        $categories=[];
        Category::getAllChildCategoryIds($Category->id,$categories);
         $log_data = [
                'model'   => 'Category',
                'action'    => 'Delete', //Create,Update,Delete
                'user_id' =>  auth()->user()->id,
                'record_id' => $Category->id,
                'label'  => $Category->name,
                'description' => '',
          ];
          Log::create($log_data);
          $parent_category_id=$Category->parent_category_id;
          $size =$Category->size;
        if ($Category->delete()) {
            Category::updateParentCategoriesSizes($parent_category_id, $size,false);
            foreach($categories as $cat){
               $delete_category= Category::findorfail($cat);
               $delete_category->delete();
            }
            event(new CategoryDeleted($Category));

            return true;
        }

        throw new GeneralException(__('exceptions.backend.categories.delete_error'));
    }
}
