<?php

namespace App\Http\Controllers\Frontend\Auth;
use Carbon\Carbon;
use App\Models\Auth\User;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use App\Http\Responses\ViewResponse;
use Illuminate\Support\Facades\View;
use App\Http\Responses\RedirectResponse;
use Illuminate\Support\Facades\Response;
use Microsoft\Graph\Graph;
use Microsoft\Graph\Model;
use League\OAuth2\Client\Provider\GenericProvider;
use League\OAuth2\Client\Token\AccessToken;
use Illuminate\Support\Facades\URL;
use Illuminate\Support\Facades\Auth;
use Illuminate\Http\Request as HttpRequest;
use App\Services\TokenRevocationService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Http;
class MicrosoftloginController extends Controller
{
    protected $provider;
    protected $tokenRevocationService;

    public function __construct(TokenRevocationService $tokenRevocationService)
    {
        $this->provider = new GenericProvider([
            'clientId'                => env('GRAPH_CLIENT_ID'),
            'clientSecret'            => env('GRAPH_CLIENT_SECRET'),
            'redirectUri'             => env('GRAPH_REDIRECT_URI'),
            'urlAuthorize'            => 'https://login.microsoftonline.com/common/oauth2/v2.0/authorize',
            'urlAccessToken'          => 'https://login.microsoftonline.com/common/oauth2/v2.0/token',
            'urlResourceOwnerDetails' => '',
            'scopes'                  => 'User.Read Mail.Send offline_access'
        ]);

        $this->tokenRevocationService = $tokenRevocationService;
    }

    public function redirectToProvider()
    {

        $authorizationUrl = $this->provider->getAuthorizationUrl([
            'prompt' => 'login'  // Force re-authentication
        ]);
        session(['state' => $this->provider->getState()]);
        return redirect()->away($authorizationUrl);
    }

    public function handleProviderCallback(Request $request)
    {
        try {
            // Try to get an access token using the authorization code grant.
            $accessToken = $this->provider->getAccessToken('authorization_code', [
                'code' => $request->code
            ]);

            // Fetch user email using the access token
            $email = $this->getUserEmail($accessToken);

            // Find user by email in the database
            $user = User::where('email', $email)->first();

            if ($user) {
                // Case 1: User wants to log into the system with Microsoft account
                if (!Auth::check()) {
                    // Log the user into the system
                    Auth::login($user);
                }

                // Case 2: User is already logged in and wants to authenticate with Microsoft
                // Store the access token details
                $this->storeTokenDetails($accessToken);

                // Redirect to frontend index
                return redirect()->route('admin.categories.index');
            } else {
                // Revoke the token and redirect to login with an error
                $this->revokeToken($accessToken);
                return redirect()->route('frontend.index')->withErrors(['error' => 'Email does not match. Access denied.']);
            }
        } catch (\Exception $e) {
            // Handle exceptions
            return redirect()->route('frontend.index')->withErrors(['error' => 'Failed to authenticate with Microsoft.']);
        }
    }

    private function storeTokenDetails($accessToken)
    {
        $expirationTimestamp = $accessToken->getExpires();
        $refreshToken = $accessToken->getRefreshToken();

        session([
            'access_token' => $accessToken->getToken(),
            'access_token_expires' => $expirationTimestamp,
            'refresh_token' => $refreshToken
        ]);

        $user = Auth::user();
        $user->microsoft_token = $refreshToken;
        $user->save();
    }

    private function getUserEmail($accessToken)
    {
        $graphUrl = 'https://graph.microsoft.com/v1.0/me';
        $response = Http::withHeaders([
            'Authorization' => 'Bearer ' . $accessToken->getToken(),
            'Accept' => 'application/json',
        ])->get($graphUrl);

        if ($response->successful()) {
            $userData = $response->json();
            return $userData['mail'] ?? $userData['userPrincipalName']; // Use 'mail' if available, otherwise fallback to 'userPrincipalName'
        } else {
            throw new \Exception('Unable to retrieve user email from Microsoft Graph.');
        }
    }

    private function revokeToken($accessToken)
    {
        $revokeUrl = 'https://login.microsoftonline.com/common/oauth2/v2.0/logout';
        Http::post($revokeUrl, [
            'token' => $accessToken->getToken(),
            'token_type_hint' => 'access_token',
            'client_id' => env('GRAPH_CLIENT_ID'),
            'client_secret' => env('GRAPH_CLIENT_SECRET')
        ]);
    }

    public function logout()
    {
        // Get the authenticated user
        $user = Auth::user();

        // Get the access token from the session
        $accessToken = session('access_token');

        // Revoke the token from the OAuth provider
        if ($accessToken) {
            $this->tokenRevocationService->revokeToken($accessToken);
        }

        // Remove the refresh token from the database
        $user->microsoft_token = null;
        $user->save();

        // Clear the session
        session()->forget(['access_token', 'access_token_expires', 'refresh_token']);

        // Redirect to the login page or any other desired page
        return redirect()->route('settings');
    }

}
